<?php
/**
 * توابع کمکی
 * Helper Functions
 */

/**
 * Escape HTML
 */
function h($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Redirect
 */
function redirect($url, $statusCode = 302) {
    header('Location: ' . $url, true, $statusCode);
    exit;
}

/**
 * دریافت URL فعلی
 */
function current_url() {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    return $protocol . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
}

/**
 * بررسی درخواست POST
 */
function is_post() {
    return $_SERVER['REQUEST_METHOD'] === 'POST';
}

/**
 * بررسی درخواست GET
 */
function is_get() {
    return $_SERVER['REQUEST_METHOD'] === 'GET';
}

/**
 * بررسی درخواست AJAX
 */
function is_ajax() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) &&
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

/**
 * دریافت پارامتر GET
 */
function get($key, $default = null) {
    return $_GET[$key] ?? $default;
}

/**
 * دریافت پارامتر POST
 */
function post($key, $default = null) {
    return $_POST[$key] ?? $default;
}

/**
 * JSON Response
 */
function json_response($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

/**
 * JSON Success Response
 */
function json_success($message, $data = []) {
    json_response([
        'success' => true,
        'message' => $message,
        'data' => $data
    ]);
}

/**
 * JSON Error Response
 */
function json_error($message, $statusCode = 400, $errors = []) {
    json_response([
        'success' => false,
        'message' => $message,
        'errors' => $errors
    ], $statusCode);
}

/**
 * فرمت تاریخ فارسی
 */
function jalali_date($timestamp = null, $format = 'Y/m/d') {
    if ($timestamp === null) {
        $timestamp = time();
    }

    if (is_string($timestamp)) {
        $timestamp = strtotime($timestamp);
    }

    // این تابع نیاز به کتابخانه تبدیل تاریخ دارد
    // اینجا فقط یک نمونه ساده است
    return date($format, $timestamp);
}

/**
 * فرمت زمان نسبی (مثلاً "2 ساعت پیش")
 */
function time_ago($datetime) {
    $timestamp = is_numeric($datetime) ? $datetime : strtotime($datetime);
    $diff = time() - $timestamp;

    if ($diff < 60) {
        return 'همین الان';
    } elseif ($diff < 3600) {
        $minutes = floor($diff / 60);
        return $minutes . ' دقیقه پیش';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' ساعت پیش';
    } elseif ($diff < 604800) {
        $days = floor($diff / 86400);
        return $days . ' روز پیش';
    } elseif ($diff < 2592000) {
        $weeks = floor($diff / 604800);
        return $weeks . ' هفته پیش';
    } else {
        return jalali_date($timestamp);
    }
}

/**
 * فرمت حجم فایل
 */
function format_bytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    return round($bytes, $precision) . ' ' . $units[$pow];
}

/**
 * فرمت عدد فارسی
 */
function persian_number($number) {
    $persian = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
    $english = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
    return str_replace($english, $persian, $number);
}

/**
 * فرمت عدد انگلیسی
 */
function english_number($number) {
    $persian = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
    $english = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
    return str_replace($persian, $english, $number);
}

/**
 * فرمت پول
 */
function format_money($amount, $currency = 'EUR') {
    $symbols = [
        'EUR' => '€',
        'USD' => '$',
        'IRR' => 'ریال'
    ];

    $symbol = $symbols[$currency] ?? $currency;
    return number_format($amount, 2) . ' ' . $symbol;
}

/**
 * تولید CSRF Token
 */
function csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * بررسی CSRF Token
 */
function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * فیلد CSRF مخفی
 */
function csrf_field() {
    return '<input type="hidden" name="csrf_token" value="' . csrf_token() . '">';
}

/**
 * اعتبارسنجی ایمیل
 */
function is_valid_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * اعتبارسنجی URL
 */
function is_valid_url($url) {
    return filter_var($url, FILTER_VALIDATE_URL) !== false;
}

/**
 * پاکسازی رشته
 */
function clean_string($string) {
    return trim(strip_tags($string));
}

/**
 * تولید Slug
 */
function slugify($string) {
    $string = trim($string);
    $string = preg_replace('/[^A-Za-z0-9-\x{0600}-\x{06FF}]+/u', '-', $string);
    $string = preg_replace('/-+/', '-', $string);
    $string = trim($string, '-');
    return strtolower($string);
}

/**
 * کوتاه کردن متن
 */
function truncate($string, $length = 100, $suffix = '...') {
    if (mb_strlen($string) <= $length) {
        return $string;
    }
    return mb_substr($string, 0, $length) . $suffix;
}

/**
 * تبدیل آرایه به HTML Options
 */
function array_to_options($array, $selected = null, $emptyOption = null) {
    $html = '';

    if ($emptyOption !== null) {
        $html .= '<option value="">' . h($emptyOption) . '</option>';
    }

    foreach ($array as $value => $label) {
        $isSelected = ($value == $selected) ? ' selected' : '';
        $html .= '<option value="' . h($value) . '"' . $isSelected . '>' . h($label) . '</option>';
    }

    return $html;
}

/**
 * دریافت نام وضعیت کاربر
 */
function get_user_status_label($status) {
    $statuses = [
        'active' => '<span class="label bg-green">فعال</span>',
        'suspended' => '<span class="label bg-orange">معلق</span>',
        'banned' => '<span class="label bg-red">مسدود</span>',
        'deleted' => '<span class="label bg-grey">حذف شده</span>'
    ];
    return $statuses[$status] ?? $status;
}

/**
 * دریافت نام نقش
 */
function get_role_label($role) {
    $roles = [
        'super_admin' => '<span class="label bg-purple">مدیر ارشد</span>',
        'admin' => '<span class="label bg-blue">مدیر</span>',
        'moderator' => '<span class="label bg-cyan">ناظر</span>'
    ];
    return $roles[$role] ?? $role;
}

/**
 * دریافت نام پلن
 */
function get_plan_name($plan, $lang = 'fa') {
    $plans = [
        'free_trial' => ['fa' => 'آزمایشی رایگان', 'en' => 'Free Trial', 'de' => 'Kostenlose Testversion'],
        'freemium' => ['fa' => 'رایگان', 'en' => 'Freemium', 'de' => 'Freemium'],
        'plus' => ['fa' => 'پلاس', 'en' => 'Plus', 'de' => 'Plus'],
        'premium' => ['fa' => 'پریمیوم', 'en' => 'Premium', 'de' => 'Premium']
    ];
    return $plans[$plan][$lang] ?? $plan;
}

/**
 * تولید رمز عبور تصادفی
 */
function generate_password($length = 12) {
    $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*';
    $password = '';
    $charsLength = strlen($chars);

    for ($i = 0; $i < $length; $i++) {
        $password .= $chars[random_int(0, $charsLength - 1)];
    }

    return $password;
}

/**
 * لاگ خطا
 */
function log_error($message, $file = 'error.log') {
    $log = date('Y-m-d H:i:s') . ' | ' . $message . PHP_EOL;
    error_log($log, 3, LOG_DIR . $file);
}

/**
 * لاگ اطلاعات
 */
function log_info($message, $file = 'info.log') {
    $log = date('Y-m-d H:i:s') . ' | ' . $message . PHP_EOL;
    error_log($log, 3, LOG_DIR . $file);
}

/**
 * دامپ متغیر (برای دیباگ)
 */
function dd(...$vars) {
    echo '<pre style="background: #1e1e1e; color: #dcdcdc; padding: 20px; direction: ltr; text-align: left;">';
    foreach ($vars as $var) {
        var_dump($var);
    }
    echo '</pre>';
    die();
}

/**
 * نمایش پیام Flash
 */
function display_flash_messages() {
    $session = Session::getInstance();
    $flashes = $session->getAllFlash();

    if (empty($flashes)) {
        return '';
    }

    $html = '';
    foreach ($flashes as $flash) {
        $type = $flash['type'] ?? 'info';
        $message = h($flash['message']);

        $html .= <<<HTML
        <div class="alert alert-{$type} alert-dismissible" role="alert">
            <button type="button" class="close" data-dismiss="alert" aria-label="بستن">
                <span aria-hidden="true">&times;</span>
            </button>
            {$message}
        </div>
        HTML;
    }

    return $html;
}

?>