<?php
/**
 * کلاس احراز هویت
 * Authentication Class
 */

class Auth {
    private static $instance = null;
    private $db;
    private $session;

    private function __construct() {
        $this->db = Database::getInstance();
        $this->session = Session::getInstance();
    }

    /**
     * دریافت نمونه یکتا (Singleton)
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * ورود ادمین
     */
    public function login($email, $password, $remember = false) {
        // بررسی تعداد تلاش‌های ناموفق
        if ($this->isLoginLocked($email)) {
            return [
                'success' => false,
                'message' => 'حساب شما به دلیل تلاش‌های ناموفق قفل شده است. لطفاً 15 دقیقه دیگر تلاش کنید.'
            ];
        }

        // یافتن کاربر
        $user = $this->db->fetchOne(
            "SELECT u.*, a.id as admin_id, a.role, a.permissions
             FROM users u
             INNER JOIN admins a ON u.id = a.user_id
             WHERE u.email = ? AND u.status = 'active'",
            [$email]
        );

        if (!$user) {
            $this->recordLoginAttempt($email, false);
            return [
                'success' => false,
                'message' => 'ایمیل یا رمز عبور اشتباه است.'
            ];
        }

        // بررسی رمز عبور
        if (!password_verify($password, $user['password_hash'])) {
            $this->recordLoginAttempt($email, false);
            return [
                'success' => false,
                'message' => 'ایمیل یا رمز عبور اشتباه است.'
            ];
        }

        // بررسی تایید ایمیل
        if (!$user['email_verified']) {
            return [
                'success' => false,
                'message' => 'لطفاً ابتدا ایمیل خود را تایید کنید.'
            ];
        }

        // ورود موفق
        $this->recordLoginAttempt($email, true);
        $this->createSession($user);
        $this->updateLastLogin($user['id']);
        $this->logAdminActivity($user['admin_id'], 'login', 'admin', $user['admin_id']);

        // Remember Me
        if ($remember) {
            $this->setRememberMeCookie($user['id']);
        }

        return [
            'success' => true,
            'message' => 'ورود موفقیت‌آمیز بود.'
        ];
    }

    /**
     * خروج ادمین
     */
    public function logout() {
        $adminId = $this->session->get('admin_id');

        if ($adminId) {
            $this->logAdminActivity($adminId, 'logout', 'admin', $adminId);
        }

        // حذف Remember Me Cookie قبل از destroy
        $this->removeRememberMeCookie();

        // پاک کردن session
        $this->session->clear();

        // تنظیم پرچم logged_out قبل از destroy
        $this->session->set('logged_out', true);

        return [
            'success' => true,
            'message' => 'خروج موفقیت‌آمیز بود.'
        ];
    }

    /**
     * بررسی ورود
     */
    public function isLoggedIn() {
        // بررسی Session
        if ($this->session->has('admin_id') && $this->session->has('user_id')) {
            // بررسی زمان انقضا
            $lastActivity = $this->session->get('last_activity', 0);
            if (time() - $lastActivity > SESSION_LIFETIME) {
                $this->logout();
                return false;
            }

            // بروزرسانی زمان فعالیت
            $this->session->set('last_activity', time());
            return true;
        }

        // بررسی Remember Me Cookie (فقط اگر session کاملاً خالی باشد)
        // این جلوگیری می‌کند از اینکه بعد از logout دوباره لاگین شود
        if (!$this->session->has('logged_out') && $this->checkRememberMeCookie()) {
            return true;
        }

        return false;
    }

    /**
     * بررسی سطح دسترسی
     */
    public function hasRole($requiredRole) {
        if (!$this->isLoggedIn()) {
            return false;
        }

        $userRole = $this->session->get('role');

        $roleHierarchy = [
            'super_admin' => 3,
            'admin' => 2,
            'moderator' => 1
        ];

        return isset($roleHierarchy[$userRole]) &&
               isset($roleHierarchy[$requiredRole]) &&
               $roleHierarchy[$userRole] >= $roleHierarchy[$requiredRole];
    }

    /**
     * بررسی مجوز خاص
     */
    public function hasPermission($permission) {
        if (!$this->isLoggedIn()) {
            return false;
        }

        // Super Admin همه مجوزها را دارد
        if ($this->session->get('role') === 'super_admin') {
            return true;
        }

        $permissions = $this->session->get('permissions', []);
        return in_array($permission, $permissions);
    }

    /**
     * دریافت اطلاعات ادمین فعلی
     */
    public function getAdmin() {
        if (!$this->isLoggedIn()) {
            return null;
        }

        return [
            'admin_id' => $this->session->get('admin_id'),
            'user_id' => $this->session->get('user_id'),
            'email' => $this->session->get('email'),
            'full_name' => $this->session->get('full_name'),
            'role' => $this->session->get('role'),
            'permissions' => $this->session->get('permissions', [])
        ];
    }

    /**
     * دریافت ID ادمین فعلی
     */
    public function getAdminId() {
        return $this->session->get('admin_id');
    }

    /**
     * دریافت ID کاربر فعلی
     */
    public function getUserId() {
        return $this->session->get('user_id');
    }

    /**
     * ایجاد Session
     */
    private function createSession($user) {
        $this->session->regenerate();
        $this->session->set('admin_id', $user['admin_id']);
        $this->session->set('user_id', $user['id']);
        $this->session->set('email', $user['email']);
        $this->session->set('full_name', $user['full_name']);
        $this->session->set('role', $user['role']);
        $this->session->set('permissions', json_decode($user['permissions'] ?? '[]', true));
        $this->session->set('last_activity', time());
        $this->session->set('ip_address', $_SERVER['REMOTE_ADDR'] ?? '');
    }

    /**
     * بروزرسانی آخرین ورود
     */
    private function updateLastLogin($userId) {
        $this->db->update(
            'users',
            [
                'last_login' => date('Y-m-d H:i:s'),
                'last_online' => date('Y-m-d H:i:s'),
                'last_ip' => $_SERVER['REMOTE_ADDR'] ?? '',
                'last_user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                'total_login_count' => $this->db->fetchColumn(
                    "SELECT total_login_count + 1 FROM users WHERE id = ?",
                    [$userId]
                )
            ],
            'id = ?',
            [$userId]
        );
    }

    /**
     * ثبت تلاش ورود
     */
    private function recordLoginAttempt($email, $success) {
        $key = 'login_attempts_' . md5($email);

        if ($success) {
            // پاک کردن تلاش‌های ناموفق
            $this->session->remove($key);
        } else {
            // افزایش تعداد تلاش‌های ناموفق
            $attempts = $this->session->get($key, []);
            $attempts[] = time();
            $this->session->set($key, $attempts);
        }
    }

    /**
     * بررسی قفل شدن ورود
     */
    private function isLoginLocked($email) {
        $key = 'login_attempts_' . md5($email);
        $attempts = $this->session->get($key, []);

        // حذف تلاش‌های قدیمی (بیش از 15 دقیقه)
        $attempts = array_filter($attempts, function($time) {
            return (time() - $time) < LOGIN_TIMEOUT;
        });

        $this->session->set($key, $attempts);

        return count($attempts) >= MAX_LOGIN_ATTEMPTS;
    }

    /**
     * تنظیم Remember Me Cookie
     */
    private function setRememberMeCookie($userId) {
        $token = bin2hex(random_bytes(32));
        $expiry = time() + (30 * 24 * 60 * 60); // 30 روز

        // ذخیره در دیتابیس
        $this->db->query(
            "INSERT INTO remember_tokens (user_id, token, expires_at) VALUES (?, ?, ?)
             ON DUPLICATE KEY UPDATE token = ?, expires_at = ?",
            [$userId, $token, date('Y-m-d H:i:s', $expiry), $token, date('Y-m-d H:i:s', $expiry)]
        );

        // تنظیم Cookie
        setcookie('remember_me', $token, $expiry, '/', '', true, true);
    }

    /**
     * بررسی Remember Me Cookie
     */
    private function checkRememberMeCookie() {
        if (!isset($_COOKIE['remember_me'])) {
            return false;
        }

        $token = $_COOKIE['remember_me'];

        $user = $this->db->fetchOne(
            "SELECT u.*, a.id as admin_id, a.role, a.permissions
             FROM users u
             INNER JOIN admins a ON u.id = a.user_id
             INNER JOIN remember_tokens rt ON u.id = rt.user_id
             WHERE rt.token = ? AND rt.expires_at > NOW() AND u.status = 'active'",
            [$token]
        );

        if ($user) {
            $this->createSession($user);
            return true;
        }

        $this->removeRememberMeCookie();
        return false;
    }

    /**
     * حذف Remember Me Cookie
     */
    private function removeRememberMeCookie() {
        if (isset($_COOKIE['remember_me'])) {
            $token = $_COOKIE['remember_me'];
            $this->db->delete('remember_tokens', 'token = ?', [$token]);

            // حذف کوکی با تنظیمات مختلف برای اطمینان
            setcookie('remember_me', '', time() - 3600, '/', '', false, true);
            setcookie('remember_me', '', time() - 3600, '/', $_SERVER['HTTP_HOST'], false, true);
            setcookie('remember_me', '', time() - 3600, '/admin/', '', false, true);

            // حذف از آرایه $_COOKIE
            unset($_COOKIE['remember_me']);
        }
    }

    /**
     * ثبت فعالیت ادمین
     */
    private function logAdminActivity($adminId, $action, $entityType = null, $entityId = null) {
        $this->db->insert('admin_activity_logs', [
            'admin_id' => $adminId,
            'action' => $action,
            'entity_type' => $entityType,
            'entity_id' => $entityId,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
    }

    /**
     * تغییر رمز عبور
     */
    public function changePassword($userId, $oldPassword, $newPassword) {
        // دریافت رمز عبور فعلی
        $currentHash = $this->db->fetchColumn(
            "SELECT password_hash FROM users WHERE id = ?",
            [$userId]
        );

        // بررسی رمز عبور قدیمی
        if (!password_verify($oldPassword, $currentHash)) {
            return [
                'success' => false,
                'message' => 'رمز عبور فعلی اشتباه است.'
            ];
        }

        // بررسی طول رمز عبور جدید
        if (strlen($newPassword) < PASSWORD_MIN_LENGTH) {
            return [
                'success' => false,
                'message' => 'رمز عبور باید حداقل ' . PASSWORD_MIN_LENGTH . ' کاراکتر باشد.'
            ];
        }

        // بروزرسانی رمز عبور
        $newHash = password_hash($newPassword, PASSWORD_BCRYPT, ['cost' => 12]);

        $this->db->update(
            'users',
            ['password_hash' => $newHash],
            'id = ?',
            [$userId]
        );

        return [
            'success' => true,
            'message' => 'رمز عبور با موفقیت تغییر کرد.'
        ];
    }
}

?>