<?php
define('AYENEH_ADMIN', true);
require_once 'php/autoload.php';

$pageTitle = 'مدیریت ادمین‌ها';
$db = Database::getInstance();
$session = Session::getInstance();
$auth = Auth::getInstance();

// فقط Super Admin می‌تواند به این صفحه دسترسی داشته باشد
if (!$auth->hasRole('super_admin')) {
    $session->setError('شما به این بخش دسترسی ندارید.');
    redirect('index.php');
}

$errors = [];

// دریافت لیست ادمین‌ها
$admins = $db->fetchAll("
    SELECT
        a.*,
        u.email,
        u.full_name,
        u.status,
        u.created_at as user_created_at,
        u.last_online,
        a.two_factor_enabled
    FROM admins a
    INNER JOIN users u ON a.user_id = u.id
    ORDER BY
        CASE a.role
            WHEN 'super_admin' THEN 1
            WHEN 'admin' THEN 2
            WHEN 'moderator' THEN 3
        END,
        a.created_at ASC
");

// پردازش افزودن ادمین جدید
if (isPost() && post('action') === 'add') {
    $userId = (int)post('user_id');
    $role = post('role');
    $permissions = post('permissions', []);

    // اعتبارسنجی
    if (!$userId) {
        $errors[] = 'شناسه کاربر الزامی است.';
    } else {
        // بررسی وجود کاربر
        $user = $db->fetchOne("SELECT id FROM users WHERE id = ?", [$userId]);
        if (!$user) {
            $errors[] = 'کاربر با این شناسه یافت نشد.';
        } else {
            // بررسی اینکه قبلاً ادمین نباشد
            $existingAdmin = $db->fetchOne("SELECT id FROM admins WHERE user_id = ?", [$userId]);
            if ($existingAdmin) {
                $errors[] = 'این کاربر قبلاً به عنوان ادمین ثبت شده است.';
            }
        }
    }

    if (!in_array($role, ['super_admin', 'admin', 'moderator'])) {
        $errors[] = 'نقش انتخاب شده نامعتبر است.';
    }

    if (empty($errors)) {
        try {
            $db->insert('admins', [
                'user_id' => $userId,
                'role' => $role,
                'permissions' => json_encode($permissions),
                'created_at' => date('Y-m-d H:i:s')
            ]);

            $session->setSuccess('ادمین جدید با موفقیت اضافه شد.');
            redirect('admins.php');

        } catch (Exception $e) {
            $errors[] = 'خطا در افزودن ادمین: ' . $e->getMessage();
            logError('Error adding admin: ' . $e->getMessage());
        }
    }
}

// پردازش تغییر نقش
if (isPost() && post('action') === 'change_role') {
    $adminId = (int)post('admin_id');
    $newRole = post('new_role');

    if (!in_array($newRole, ['super_admin', 'admin', 'moderator'])) {
        $errors[] = 'نقش انتخاب شده نامعتبر است.';
    }

    // نمی‌توان نقش خودش را تغییر داد
    $admin = $db->fetchOne("SELECT user_id FROM admins WHERE id = ?", [$adminId]);
    if ($admin && $admin['user_id'] == $auth->getUserId()) {
        $errors[] = 'شما نمی‌توانید نقش خودتان را تغییر دهید.';
    }

    if (empty($errors)) {
        try {
            $db->update('admins', [
                'role' => $newRole,
                'updated_at' => date('Y-m-d H:i:s')
            ], 'id = ?', [$adminId]);

            $session->setSuccess('نقش ادمین با موفقیت تغییر کرد.');
            redirect('admins.php');

        } catch (Exception $e) {
            $errors[] = 'خطا در تغییر نقش: ' . $e->getMessage();
        }
    }
}

include 'php/includes/header.php';
?>


<div class="row clearfix">
    <div class="col-lg-12">
        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger">
                <strong>خطاها:</strong>
                <ul class="m-b-0">
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo h($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <!-- کارت افزودن ادمین جدید -->
        <div class="card">
            <div class="header bg-green">
                <h2 class="text-white">
                    <i class="material-icons">person_add</i>
                    افزودن ادمین جدید
                </h2>
            </div>
            <div class="body">
                <form method="POST" action="" class="form-inline">
                    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                    <input type="hidden" name="action" value="add">

                    <div class="form-group m-r-15">
                        <label class="m-l-10">شناسه کاربر (User ID):</label>
                        <input type="number" name="user_id" class="form-control" required
                               placeholder="مثلاً: 5" min="1">
                    </div>

                    <div class="form-group m-r-15">
                        <label class="m-l-10">نقش:</label>
                        <select name="role" class="form-control" required>
                            <option value="moderator">Moderator (ناظر)</option>
                            <option value="admin">Admin (مدیر)</option>
                            <option value="super_admin">Super Admin (مدیر ارشد)</option>
                        </select>
                    </div>

                    <button type="submit" class="btn-action btn-save waves-effect">
                        <i class="material-icons">add</i>
                        <span>افزودن ادمین</span>
                    </button>
                </form>

                <div class="alert alert-info m-t-15 m-b-0">
                    <i class="material-icons">info</i>
                    برای افزودن ادمین، ابتدا باید کاربر در سیستم ثبت‌نام کرده باشد. سپس شماره ID آن کاربر را وارد کنید.
                </div>
            </div>
        </div>

        <!-- لیست ادمین‌ها -->
        <div class="card">
            <div class="header">
                <h2>
                    لیست ادمین‌ها
                    <small><?php echo count($admins); ?> ادمین</small>
                </h2>
            </div>
            <div class="body">
                <div class="table-responsive">
                    <table class="table table-bordered table-striped table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>نام / ایمیل</th>
                                <th>نقش</th>
                                <th>وضعیت</th>
                                <th>2FA</th>
                                <th>تاریخ ایجاد</th>
                                <th>آخرین آنلاین</th>
                                <th>عملیات</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($admins as $admin): ?>
                            <tr>
                                <td><?php echo $admin['user_id']; ?></td>
                                <td>
                                    <strong><?php echo h($admin['full_name'] ?? 'نامشخص'); ?></strong>
                                    <br>
                                    <small class="text-muted"><?php echo h($admin['email']); ?></small>
                                </td>
                                <td>
                                    <?php
                                    $roleBadges = [
                                        'super_admin' => '<span class="badge bg-purple">مدیر ارشد</span>',
                                        'admin' => '<span class="badge bg-blue">مدیر</span>',
                                        'moderator' => '<span class="badge bg-cyan">ناظر</span>'
                                    ];
                                    echo $roleBadges[$admin['role']] ?? $admin['role'];
                                    ?>
                                </td>
                                <td><?php echo getUserStatusBadge($admin['status']); ?></td>
                                <td>
                                    <?php if ($admin['two_factor_enabled']): ?>
                                        <span class="badge bg-green" title="فعال">
                                            <i class="material-icons" style="font-size: 16px; vertical-align: middle;">verified_user</i>
                                            فعال
                                        </span>
                                    <?php else: ?>
                                        <span class="badge bg-grey" title="غیرفعال">
                                            <i class="material-icons" style="font-size: 16px; vertical-align: middle;">lock_open</i>
                                            غیرفعال
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo formatDate($admin['created_at']); ?></td>
                                <td>
                                    <?php echo $admin['last_online'] ? timeAgo($admin['last_online']) : '-'; ?>
                                </td>
                                <td class="table-actions">
                                    <!-- دکمه فعال/غیرفعال 2FA - همه ادمین‌ها می‌توانند استفاده کنند -->
                                    <?php if ($admin['two_factor_enabled']): ?>
                                        <button class="btn btn-xs btn-warning"
                                                onclick="toggle2FA(<?php echo $admin['id']; ?>, 'disable', '<?php echo h($admin['full_name'] ?? $admin['email']); ?>')"
                                                title="غیرفعال کردن 2FA">
                                            <i class="material-icons">lock_open</i>
                                        </button>
                                    <?php else: ?>
                                        <button class="btn btn-xs btn-success"
                                                onclick="toggle2FA(<?php echo $admin['id']; ?>, 'enable', '<?php echo h($admin['full_name'] ?? $admin['email']); ?>')"
                                                title="فعال کردن 2FA">
                                            <i class="material-icons">verified_user</i>
                                        </button>
                                    <?php endif; ?>

                                    <?php if ($admin['user_id'] != $auth->getUserId()): ?>
                                        <!-- دکمه تغییر نقش -->
                                        <div class="btn-group">
                                            <button type="button" class="btn btn-xs btn-primary dropdown-toggle"
                                                    data-toggle="dropdown">
                                                <i class="material-icons">swap_horiz</i>
                                                تغییر نقش
                                            </button>
                                            <ul class="dropdown-menu">
                                                <?php if ($admin['role'] !== 'super_admin'): ?>
                                                <li>
                                                    <a href="#" onclick="changeRole(<?php echo $admin['id']; ?>, 'super_admin'); return false;">
                                                        مدیر ارشد
                                                    </a>
                                                </li>
                                                <?php endif; ?>

                                                <?php if ($admin['role'] !== 'admin'): ?>
                                                <li>
                                                    <a href="#" onclick="changeRole(<?php echo $admin['id']; ?>, 'admin'); return false;">
                                                        مدیر
                                                    </a>
                                                </li>
                                                <?php endif; ?>

                                                <?php if ($admin['role'] !== 'moderator'): ?>
                                                <li>
                                                    <a href="#" onclick="changeRole(<?php echo $admin['id']; ?>, 'moderator'); return false;">
                                                        ناظر
                                                    </a>
                                                </li>
                                                <?php endif; ?>
                                            </ul>
                                        </div>

                                        <!-- دکمه حذف -->
                                        <button class="btn btn-xs btn-danger"
                                                onclick="removeAdmin(<?php echo $admin['id']; ?>, '<?php echo h($admin['full_name'] ?? $admin['email']); ?>')">
                                            <i class="material-icons">delete</i>
                                        </button>
                                    <?php else: ?>
                                        <span class="badge bg-grey">شما</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- توضیحات نقش‌ها -->
        <div class="card">
            <div class="header">
                <h2>توضیحات نقش‌ها</h2>
            </div>
            <div class="body">
                <div class="row">
                    <div class="col-md-4">
                        <div class="alert alert-info">
                            <h4><span class="badge bg-purple">Super Admin</span> مدیر ارشد</h4>
                            <ul class="m-b-0">
                                <li>دسترسی کامل به تمام بخش‌ها</li>
                                <li>مدیریت ادمین‌ها</li>
                                <li>تغییر تنظیمات سیستم</li>
                                <li>مشاهده و تغییر همه چیز</li>
                            </ul>
                        </div>
                    </div>

                    <div class="col-md-4">
                        <div class="alert alert-success">
                            <h4><span class="badge bg-blue">Admin</span> مدیر</h4>
                            <ul class="m-b-0">
                                <li>مدیریت کاربران</li>
                                <li>مدیریت محتوا</li>
                                <li>ارسال اعلان‌ها</li>
                                <li>مشاهده گزارش‌ها</li>
                            </ul>
                        </div>
                    </div>

                    <div class="col-md-4">
                        <div class="alert alert-warning">
                            <h4><span class="badge bg-cyan">Moderator</span> ناظر</h4>
                            <ul class="m-b-0">
                                <li>مشاهده کاربران</li>
                                <li>Ban/Unban کاربران</li>
                                <li>مشاهده گزارش‌های محدود</li>
                                <li>پاسخ به تیکت‌ها</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- فرم مخفی برای تغییر نقش -->
<form id="changeRoleForm" method="POST" action="" style="display: none;">
    <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
    <input type="hidden" name="action" value="change_role">
    <input type="hidden" name="admin_id" id="changeRoleAdminId">
    <input type="hidden" name="new_role" id="changeRoleNewRole">
</form>

<script>
function changeRole(adminId, newRole) {
    if (confirm('آیا از تغییر نقش این ادمین اطمینان دارید؟')) {
        document.getElementById('changeRoleAdminId').value = adminId;
        document.getElementById('changeRoleNewRole').value = newRole;
        document.getElementById('changeRoleForm').submit();
    }
}

function removeAdmin(adminId, adminName) {
    if (confirm('آیا از حذف "' + adminName + '" از لیست ادمین‌ها اطمینان دارید؟\n\nتوجه: این کاربر همچنان در سیستم باقی می‌ماند، فقط دسترسی ادمین آن حذف می‌شود.')) {
        window.location.href = 'api/admin_remove.php?id=' + adminId;
    }
}

function toggle2FA(adminId, action, adminName) {
    let message = '';
    if (action === 'enable') {
        message = 'آیا می‌خواهید 2FA را برای "' + adminName + '" فعال کنید؟\n\nبعد از فعال‌سازی، یک QR Code برای تنظیم Google Authenticator نمایش داده می‌شود.';
    } else {
        message = 'آیا می‌خواهید 2FA را برای "' + adminName + '" غیرفعال کنید؟\n\nتوجه: این کار امنیت حساب را کاهش می‌دهد.';
    }

    if (confirm(message)) {
        window.location.href = 'api/admin_toggle_2fa.php?id=' + adminId + '&action=' + action;
    }
}
</script>

<?php include 'php/includes/footer.php'; ?>
